<?php
/* ======================================================
 # Fix 404 Error Links for Joomla! - v2.2.5 (pro version)
 # -------------------------------------------------------
 # For Joomla! CMS (v3.x)
 # Author: Web357 (Yiannis Christodoulou)
 # Copyright (©) 2014-2022 Web357. All rights reserved.
 # License: GNU/GPLv3, http://www.gnu.org/licenses/gpl-3.0.html
 # Website: https:/www.web357.com
 # Demo: https://demo.web357.com/joomla/fix-404-error-links
 # Support: support@web357.com
 # Last modified: Thursday 08 June 2023, 01:53:23 AM
 ========================================================= */
defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\Utilities\IpHelper;
use Joomla\CMS\Application\CMSApplication;

class plgSystemFix404ErrorLinks extends JPlugin
{	
	private static $live_url;
	private static $current_url;
	private static $current_url_path;
	private static $user_id;
	private static $sql_datetime;
	private static $joomla_datetime;
	private static $ip_address;
	private static $user_agent;
	private static $is_bot;
	private static $referrer;
	private static $country;
	private static $country_name;
	private static $country_code;
	private static $continent_code;
	private static $browser;
	private static $operating_system;

	public function __construct(&$subject, $config)
	{
		parent::__construct($subject, $config);

		JLog::addLogger(array('text_file' => 'com_fix404errorlinks.log.php'), JLog::ALL, array('com_fix404errorlinks'));

		$app = Factory::getApplication();
		if ($app->isClient('site'))
		{		
			// Get URL details
			$uri = JUri::getInstance();
			self::$live_url = JURI::base();
			self::$current_url = rawurldecode($uri->toString(array('scheme', 'host', 'port', 'path', 'query', 'fragment')));
			$current_url_path = str_replace(JURI::base(), '', self::$current_url); // without domain
			$current_url_path = self::idnToAscii($current_url_path);
			self::$current_url_path = htmlentities($current_url_path, ENT_COMPAT, 'UTF-8');

			// Get User details
			self::$user_id = Factory::getUser()->get('id');

			// Get datetime
			$jdate = new JDate;
			self::$sql_datetime = $jdate->toSql();
			self::$joomla_datetime = JHtml::_('date', self::$sql_datetime, JText::_('DATE_FORMAT_LC6'));

			// Get the correct IP address of the client
			$ip_address = IpHelper::getIp();
			if (!filter_var($ip_address, FILTER_VALIDATE_IP))
			{
				$ip_address = isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '0.0.0.0';
			}
			self::$ip_address = $ip_address;
			// self::$ip_address = '66.102.15.255'; // googlebottest
			// self::$ip_address = '41.175.240.46'; // proxytest

			// Get Agent
			self::$user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : ''; // Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)
			//self::$user_agent = 'Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)';
			//self::$user_agent = 'Mozilla/5.0 (compatible; DotBot/1.1; http://www.opensiteexplorer.org/dotbot, help@moz.com)';
			//self::$user_agent = 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.181 Safari/537.36';

			// Check if is bot
			self::$is_bot = (strpos(self::$user_agent, 'bot') !== false) ? true : false; // returns true if is a bot, or false if is not a bot.
			
			// Get Referrer
			self::$referrer = (!empty($_SERVER['HTTP_REFERER'])) ? $_SERVER['HTTP_REFERER'] : ''; // HTTP Referer

			// Get Geo/Browser/OS information
			self::$country = $this->getDataFromGeoIP('country_name').' ('.$this->getDataFromGeoIP('continent').')'; // Greece (EU)
			self::$country_name = $this->getDataFromGeoIP('country_name'); // Greece
			self::$country_code = $this->getDataFromGeoIP('country_code'); // GR
			self::$continent_code = $this->getDataFromGeoIP('continent'); // EU
			self::$browser = $this->getBrowser(); // Web Browser
			self::$operating_system = $this->getOS(); // Operating System		
		}
	}

	public static function getRedirects()
	{
		$db = Factory::getDBO();
		$query = $db->getQuery(true);
		$query->select($db->quoteName(array('source_url', 'target_url', 'action_code', 'regex', 'match_type', 'source_match_http_code', 'action_type')));
		$query->from($db->quoteName('#__fix404errorlinks_redirects'));
		$query->where($db->quoteName('state').' = 1');
		$query->order($db->quoteName('ordering').' ASC');
		$db->setQuery($query);

		try
		{
			return $db->loadObjectList();
		}
		catch (RuntimeException $e)
		{
			JError::raiseError(500, $e->getMessage());
			return false;
		}
	}

	private static function onAfterInitialise_onAfterRoute()
	{
		$app = Factory::getApplication();
		
		// get the Joomla! version
		if (version_compare(JVERSION, '3.0', 'lt'))
		{
			// The plugin is not working in Joomla! 2.5.
			return;
		}
		
		// Make sure we are not in the administrator
		if ($app->isClient('administrator'))
		{
			return;
		}

        if (version_compare(JVERSION, '4.0', '<')) {
            // Set the error handler for E_ERROR to be the class handleError method.
            JError::setErrorHandling(E_ERROR, 'callback', array('plgSystemFix404ErrorLinks', 'web357HandleError404'));
            set_exception_handler(array('plgSystemFix404ErrorLinks', 'web357HandleError404'));
        }

		self::redirectFixedURLs();
	}

	public function onAfterInitialise()
	{
		self::onAfterInitialise_onAfterRoute();
	}
	
	public function onAfterRoute()
	{
		self::onAfterInitialise_onAfterRoute();
	}

	public function onError($error)
	{
		$app = JFactory::getApplication();
		
		// get the Joomla! version
		if (version_compare(JVERSION, '3.0', 'lt'))
		{
			// The plugin is not working in Joomla! 2.5.
			return;
		}
		
		// Make sure we are not in the administrator
		if ($app->isClient('administrator'))
		{
			return;
		}

		self::web357HandleError404($error);
		self::redirectFixedURLs();
	}

	public static function redirectFixedURLs()
	{
		// get fixed urls from redirects sql table
		$redirects = self::getRedirects();

		foreach ($redirects as $redirect)
		{
			$redirect_source_url = self::idnToAscii($redirect->source_url);
			$redirect_source_url = htmlentities($redirect_source_url, ENT_COMPAT, 'UTF-8');

			switch ($redirect->match_type)
			{
				case "full_url":

					$pattern_source_url = "/".$redirect_source_url."/";
					if ($redirect->regex == TRUE && preg_match($pattern_source_url, self::$current_url_path))
					{
						self::redirectToTheFinalDestination($redirect->target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);	
					}
					else if ($redirect->regex != TRUE && self::$current_url_path == $redirect_source_url)
					{
						if(self::$current_url_path == $redirect->target_url)
						{
							return;
						}

						self::redirectToTheFinalDestination($redirect->target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);		
					}

				break;
				case "contains_string":
					
					$pattern_source_url = "/".$redirect_source_url."/";
					if ($redirect->regex == TRUE && preg_match($pattern_source_url, self::$current_url_path))
					{
						self::redirectToTheFinalDestination($redirect->target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);	
					}
					else if ($redirect->regex != TRUE && strpos(self::$current_url_path, $redirect_source_url) !== false)
					{
						if(self::$current_url_path == $redirect->target_url)
						{
							return;
						}
						self::redirectToTheFinalDestination($redirect->target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);		
					}

				break;
				case "start_with":

					$pattern_source_url = "/".$redirect_source_url."/";
					if ($redirect->regex == TRUE && preg_match($pattern_source_url, self::$current_url_path))
					{
						self::redirectToTheFinalDestination($redirect->target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);	
					}
					else if ($redirect->regex != TRUE && self::startsWith(self::$current_url_path, $redirect_source_url))
					{
						self::redirectToTheFinalDestination($redirect->target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);		
					}

				break;
				case "end_with":

					$pattern_source_url = "/".$redirect_source_url."/";
					if ($redirect->regex == TRUE && preg_match($pattern_source_url, self::$current_url_path))
					{
						self::redirectToTheFinalDestination($redirect->target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);	
					}
					else if ($redirect->regex != TRUE && self::endsWith(self::$current_url_path, $redirect_source_url))
					{
						self::redirectToTheFinalDestination($redirect->target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);		
					}

				break;
				case "find_and_replace":
					$pattern_source_url = "/".$redirect_source_url."/";
					if ($redirect->regex == TRUE && preg_match($pattern_source_url, self::$current_url_path))
					{
						// find a string in error url and replace that string with new string
						$target_url = preg_replace($pattern_source_url, $redirect->target_url, self::$current_url_path); 
						self::redirectToTheFinalDestination($target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);	
					}
					else if ($redirect->regex != TRUE && strpos(self::$current_url_path, $redirect_source_url) !== false)
					{
						// find a string in error url and replace that string with new string
						$target_url = str_replace($redirect_source_url, $redirect->target_url, self::$current_url_path); 
						if(strpos(self::$current_url_path, $redirect->target_url) !== false)
						{
							return;
						}
						self::redirectToTheFinalDestination($target_url, $redirect->source_url, $redirect->source_match_http_code, $redirect->regex, $redirect->match_type, $redirect->action_type, $redirect->action_code);		
					}
				break;
			}
		}
	}

	public static function redirectToTheFinalDestination($target_url = '', $source_url = '', $source_match_http_code = 404, $regex = 0, $match_type = 'full_url', $action_type = 'redirect_to_url', $action_code = 301)
	{
		if (empty($source_url))
		{
			$source_url = JURI::base();
		}

		if (empty($target_url))
		{
			$target_url = JURI::base();
		}

		if (empty($action_code))
		{
			$action_code = 301;
		}

		# if (!$source_match_http_code)
		# {
		# 	$source_match_http_code = 404;
		# }
# 
		# // if the http status code of current url is not matched with the given source_match_http_code do not redirect the user to the target page
		# if (self::getUrlHttpCode(self::$current_url) != $source_match_http_code)
		# {
		# 	//JLog::add('do not redirect because the status code of the current and source url are not', JLog::DEBUG, 'com_fix404errorlinks');
		# 	return;
		# }
		# else
		# {
		# 	//JLog::add('redirect because the status code of the current and source url are the same', JLog::DEBUG, 'com_fix404errorlinks');
		# }

		/**
		 * A map of integer HTTP 1.1 response codes to the full HTTP Status for the headers.
		 *
		 * @var    object
		 * @since  3.4
		 * @link   http://www.iana.org/assignments/http-status-codes/
		 */
		$http_status_code_arr = array(
			100 => 'HTTP/1.1 100 Continue',
			101 => 'HTTP/1.1 101 Switching Protocols',
			102 => 'HTTP/1.1 102 Processing',
			200 => 'HTTP/1.1 200 OK',
			201 => 'HTTP/1.1 201 Created',
			202 => 'HTTP/1.1 202 Accepted',
			203 => 'HTTP/1.1 203 Non-Authoritative Information',
			204 => 'HTTP/1.1 204 No Content',
			205 => 'HTTP/1.1 205 Reset Content',
			206 => 'HTTP/1.1 206 Partial Content',
			207 => 'HTTP/1.1 207 Multi-Status',
			208 => 'HTTP/1.1 208 Already Reported',
			226 => 'HTTP/1.1 226 IM Used',
			300 => 'HTTP/1.1 300 Multiple Choices',
			301 => 'HTTP/1.1 301 Moved Permanently',
			302 => 'HTTP/1.1 302 Found',
			303 => 'HTTP/1.1 303 See other',
			304 => 'HTTP/1.1 304 Not Modified',
			305 => 'HTTP/1.1 305 Use Proxy',
			306 => 'HTTP/1.1 306 (Unused)',
			307 => 'HTTP/1.1 307 Temporary Redirect',
			308 => 'HTTP/1.1 308 Permanent Redirect',
			400 => 'HTTP/1.1 400 Bad Request',
			401 => 'HTTP/1.1 401 Unauthorized',
			402 => 'HTTP/1.1 402 Payment Required',
			403 => 'HTTP/1.1 403 Forbidden',
			404 => 'HTTP/1.1 404 Not Found',
			405 => 'HTTP/1.1 405 Method Not Allowed',
			406 => 'HTTP/1.1 406 Not Acceptable',
			407 => 'HTTP/1.1 407 Proxy Authentication Required',
			408 => 'HTTP/1.1 408 Request Timeout',
			409 => 'HTTP/1.1 409 Conflict',
			410 => 'HTTP/1.1 410 Gone',
			411 => 'HTTP/1.1 411 Length Required',
			412 => 'HTTP/1.1 412 Precondition Failed',
			413 => 'HTTP/1.1 413 Payload Too Large',
			414 => 'HTTP/1.1 414 URI Too Long',
			415 => 'HTTP/1.1 415 Unsupported Media Type',
			416 => 'HTTP/1.1 416 Requested Range Not Satisfiable',
			417 => 'HTTP/1.1 417 Expectation Failed',
			418 => 'HTTP/1.1 418 I\'m a teapot',
			422 => 'HTTP/1.1 422 Unprocessable Entity',
			423 => 'HTTP/1.1 423 Locked',
			424 => 'HTTP/1.1 424 Failed Dependency',
			425 => 'HTTP/1.1 425 Reserved for WebDAV advanced collections expired proposal',
			426 => 'HTTP/1.1 426 Upgrade Required',
			428 => 'HTTP/1.1 428 Precondition Required',
			429 => 'HTTP/1.1 429 Too Many Requests',
			431 => 'HTTP/1.1 431 Request Header Fields Too Large',
			451 => 'HTTP/1.1 451 Unavailable For Legal Reasons',
			500 => 'HTTP/1.1 500 Internal Server Error',
			501 => 'HTTP/1.1 501 Not Implemented',
			502 => 'HTTP/1.1 502 Bad Gateway',
			503 => 'HTTP/1.1 503 Service Unavailable',
			504 => 'HTTP/1.1 504 Gateway Timeout',
			505 => 'HTTP/1.1 505 HTTP Version Not Supported',
			506 => 'HTTP/1.1 506 Variant Also Negotiates (Experimental)',
			507 => 'HTTP/1.1 507 Insufficient Storage',
			508 => 'HTTP/1.1 508 Loop Detected',
			510 => 'HTTP/1.1 510 Not Extended',
			511 => 'HTTP/1.1 511 Network Authentication Required',
		);

		switch ($action_type) 
		{
			case 'redirect_to_url':
				self::increaseHits($source_url, $target_url, $regex, $match_type, $action_type, $action_code);
				
				$target_url_log = ltrim(str_replace(JURI::base(), '', $target_url), '/');
				$base_url = JRoute::_(JURI::base(), true, true);
				header($http_status_code_arr[$action_code]);
				if (strpos($target_url, 'http://') !== false || strpos($target_url, 'https://') !== false)
				{
					header('Location: ' . $target_url);
				}
				else
				{
					header('Location: ' . $base_url . '' .$target_url);
				}
				jexit();

			break;

			case 'redirect_to_menu_item':
				$target_menu_item_id = intval($target_url);
				self::increaseHits($source_url, $target_menu_item_id, $regex, $match_type, $action_type, $action_code, $target_menu_item_id);
				$router = CMSApplication::getInstance('site')->getRouter('site');
				$url = $router->build('index.php?Itemid=' . intval($target_url));
				$target_url = $url->toString();
				$target_url_log = ltrim(str_replace(JURI::base(true), '', $target_url), '/');
				header($http_status_code_arr[$action_code]);
				header('Location: ' . $target_url);
				jexit();

			break;

			case 'error_404':

				$action_code = (substr($action_code, 0, 1) != 4) ? 404 : $action_code;
				self::increaseHits($source_url, '', $regex, $match_type, $action_type, $action_code);

				for ($i=0; $i<count($http_status_code_arr); $i++)
				{
					if (array_key_exists($action_code, $http_status_code_arr))
					{
						$http_status = preg_replace('/HTTP\/1.1\s[0-9]{3}\s/i', '', $http_status_code_arr[$action_code]);
						throw new Exception(JText::_($http_status), $action_code);
					}
				}
			break;
			
			case 'do_nothing':
				self::increaseHits($source_url, '', $regex, $match_type, $action_type, $action_code);
			break;
		}
	}

	public static function increaseHits($source_url = '', $target_url = '', $regex = 0, $match_type = 'full_url', $action_type = 'redirect_to_url', $action_code = 301, $target_menu_item_id = 0)
	{
		// Database connection
		$db = Factory::getDBO();

		// Increase hits
		$query = $db->getQuery(true);
		$query->select('*');
		$query->from($db->quoteName('#__fix404errorlinks_redirects'));
		
		switch ($action_type) 
		{
			case 'redirect_to_url':
				
				if ($regex || ($match_type == 'contains_string'))
				{
					$query->where($db->quote(self::$current_url_path)." REGEXP ".$db->quote($source_url));
				}
				else
				{
					$query->where($db->quoteName('source_url')." = ".$db->quote(self::$current_url_path));
				}

			break;

			case 'redirect_to_menu_item':
				
				if ($regex || ($match_type == 'contains_string'))
				{
					$query->where($db->quote(self::$current_url_path)." REGEXP ".$db->quote($source_url));
				}
				else
				{
					$query->where($db->quoteName('source_url')." = ".$db->quote(self::$current_url_path));
				}

			break;

			case 'error_404':

				if ($regex || ($match_type == 'contains_string'))
				{
					$query->where($db->quote(self::$current_url_path)." REGEXP ".$db->quote($source_url));
				}
				else
				{
					$query->where($db->quoteName('source_url')." = ".$db->quote(self::$current_url_path));
				}

			break;
			
			case 'do_nothing':

				if ($regex || ($match_type == 'contains_string'))
				{
					$query->where($db->quote(self::$current_url_path)." REGEXP ".$db->quote($source_url));
				}
				else
				{
					$query->where($db->quoteName('source_url')." = ".$db->quote(self::$current_url_path));
				}

			break;
		}

		$query->where($db->quoteName('regex')." = ".(int) $regex);
		$query->where($db->quoteName('match_type')." = ".$db->quote($match_type));
		$query->where($db->quoteName('action_type')." = ".$db->quote($action_type));
		$query->where($db->quoteName('state')." = 1");
		$db->setQuery($query);
		$redirect = $db->loadObject();

		if (empty($redirect))
		{
			return;
		}

		try
		{
			$redirect->hits++;
			$redirect->last_access = self::$sql_datetime;
			$db->updateObject('#__fix404errorlinks_redirects', $redirect, array('source_url', 'id'));
		}
		catch (Exception $e)
		{
			JErrorPage::render(new Exception(JText::_('Fix404ErrorLinks: Error Updating Database.'), 500, $e));
		}
	}

	public static function web357HandleError404($error)
	{
		// Get the application object.
		$app = Factory::getApplication();
		
		if ($app->isClient('administrator'))
		{
			return;
		}

		if (empty(self::$current_url_path))
		{
			return;
		}

		// Load language file
		$lang = Factory::getLanguage();
		$extension = 'com_fix404errorlinks';
		$base_dir = JPATH_ADMINISTRATOR . '/components/' . $extension;
		$lang->load($extension, $base_dir);

		// get vars
		$comParams = JComponentHelper::getParams('com_fix404errorlinks');
		$config = Factory::getConfig();
		$option = $app->input->get('option', '', 'STRING');
		
		// Logging
		$enable_logging = $comParams->get('enable_logging', 1);

		// Email Settings
		$email_notifications_from_guests = $comParams->get('email_notifications_from_guests', 1);
		$email_notifications_from_bots = $comParams->get('email_notifications_from_bots', 0);
		$email_notifications_from_danger_strings = $comParams->get('email_notifications_from_danger_strings', 1);

		// 400 (Bad request), 401 (Authorization required), 403 (Forbidden), 404 (Not found), 500 (Internal server error), 1146 (SQL Error)
		if ($error->getCode() || $error->getCode() === 0)
		{
			// Define vars
			$is_bot_ip_check = false;
			$enable_bot_logging = false;

			// Get Component Parameters
			$danger_strings_logging = $comParams->get('danger_strings_logging', 1);
			$danger_strings_list_default_value = implode("\n", self::dangerStrings());
			$danger_strings_list = $comParams->get('dangerstringslist', $danger_strings_list_default_value);
			$danger_strings_list_arr = explode("\n", $danger_strings_list);
			$danger_strings_redirection_type = $comParams->get('danger_strings_redirection_type', 'url');
			$danger_url_redirects_to_url = $comParams->get('danger_url_redirects_to_url', 'index.php');
			$danger_url_redirects_to_menuitem = $comParams->get('danger_url_redirects_to_menuitem', '');
			$exclude_continents = $comParams->get('exclude_continents', array());
			$exclude_countries = $comParams->get('exclude_countries', array());
			$redirect_all_404errors_in_one_page_redirection_type = $comParams->get('redirect_all_404errors_in_one_page_redirection_type', 'url');
			
			$danger_strings_redirection = $comParams->get('danger_strings_redirection', 1);
			$redirect_all_404errors_in_one_page = $comParams->get('redirect_all_404errors_in_one_page', 0);

			$all_404_error_urls_redirects_to_url = $comParams->get('all_404_error_urls_redirects_to_url', 'index.php');
			$all_404_error_urls_redirects_to_menuitem = $comParams->get('all_404_error_urls_redirects_to_menuitem', '');

			$enable_bot_logging_param = $comParams->get('enable_bot_logging', 0);

			// Database connection
			$db = Factory::getDBO();
				
			// Get Error details
			$get_error_code = $error->getCode(); // 400, 401, 403, 404, 500
			$get_error_message = htmlspecialchars($error->getMessage(), ENT_QUOTES, 'UTF-8'); // error message
			$get_error_message = preg_replace("/\([^)]+\)/", "", $get_error_message); // error message without parenthesis url

			$is_danger_txt = '';
			$danger_string = '';

			if (self::$is_bot) // check if this IP is bot
			{
				$is_bot_ip_check = true;

				if (!$enable_bot_logging_param)
				{
					// is bot, but do not store this log into the database.
					$enable_bot_logging = false;
				}
				elseif ($enable_bot_logging_param)
				{
					// is bot, store this log into the database.
					$enable_bot_logging = true;
				}
			}

			// Avoid some folders
			if (strpos(self::$current_url_path, "cache/") !== false)
			{
				$app->redirect(JRoute::_('index.php', false));
			}
				
			// Check if the error url is included in the danger strings list array
			$is_danger = false;
			$is_danger_txt = '';
			if (!empty($danger_strings_list_arr))
			{
				foreach ($danger_strings_list_arr as $danger_string)
				{
					$danger_string = trim($danger_string);
					if (!empty($danger_string) && strpos(self::$current_url_path, $danger_string) !== false) // check if danger string exists on the current url
					{
						$is_danger = true;
						$is_danger_txt = '<span style="color:red;">'.JText::_('COM_FIX404ERRORLINKS_DANGER').'</span>';
					}
				}
			}

			// Get data from the current url
			$query = $db->getQuery(true);
			$query->select('*');
			$query->from($db->quoteName('#__fix404errorlinks_error404logs'));
			$query->where($db->quoteName('error_url')." = ".$db->quote(self::$current_url_path));
			$db->setQuery($query);
			$error_url = $db->loadObject();

			// Prepare the insert query.
			if (
				in_array(self::$continent_code, $exclude_continents) == FALSE && // Exclude Continents
				in_array(self::$country_code, $exclude_countries) == FALSE && // Exclude Countries
				(($enable_logging == TRUE && (self::$is_bot == FALSE && $is_danger == FALSE)) || // guests
				($danger_strings_logging == TRUE && $enable_bot_logging == TRUE && (self::$is_bot == TRUE && $is_danger == TRUE)) || // bot which includes danger string
				($danger_strings_logging == TRUE && (self::$is_bot == FALSE && $is_danger == TRUE)) || // danger strings
				($enable_bot_logging == TRUE && (self::$is_bot == TRUE && $is_danger == FALSE))) // bots
				)
			{
				if ($error_url === null)
				{
					// Store the data into the database
					$error_url = new stdClass();
					$error_url->error_url = self::$current_url_path;
					$error_url->error_code = $get_error_code;
					$error_url->error_message = $get_error_message;
					$error_url->datetime_created = self::$sql_datetime;
					$error_url->state = 1;
					$error_url->hits = 1;
					$error_url->mailsent = 0;

					// vars for email
					$hits_count = 1;
					$hits_error_url = $error_url->error_url;
					
					$db = Factory::getDbo();

					try
					{
						$db->insertObject('#__fix404errorlinks_error404logs', $error_url, 'id');
					}
					catch (Exception $e)
					{
						JErrorPage::render(new Exception(JText::_('Fix404ErrorLinks: Error Updating Database.'), 500, $e));
					}
				}
				else
				{
					$error_url->hits++;
					$error_url->datetime_created = self::$sql_datetime;

					// vars for email
					$hits_count = $error_url->hits;
					$hits_error_url = $error_url->error_url;

					try
					{
						$db->updateObject('#__fix404errorlinks_error404logs', $error_url, 'error_url');
					}
					catch (Exception $e)
					{
						JErrorPage::render(new Exception(JText::_('Fix404ErrorLinks: Error Updating Database.'), 500, $e));
					}
				}
			}

			// mail subject
			if(self::$is_bot)
			{
				$subject = JText::sprintf(JText::_('COM_FIX404ERRORLINKS_EMAIL_SUBJECT_BOT'), $get_error_code, $get_error_message, self::$ip_address, self::$country);
			}
			elseif($is_danger)
			{
				$subject = JText::sprintf(JText::_('COM_FIX404ERRORLINKS_EMAIL_SUBJECT_DANGER'), $get_error_code, $get_error_message, self::$ip_address, self::$country);
			}
			else
			{ 
				$subject = JText::sprintf(JText::_('COM_FIX404ERRORLINKS_EMAIL_SUBJECT_GUEST'), $get_error_code, $get_error_message, self::$ip_address, self::$country);
			}

			// Variables for Email
			$mail_subject = JText::_('COM_FIX404ERRORLINKS_ERROR')." ".$get_error_code.": ".$get_error_message;
			$mail_error_url_link = '<a href="'.self::$current_url.'" title="'.self::$current_url.'" target="_blank">'.self::$current_url.'</a>';
			$mail_error_url_txt = '<strong>'.JText::_('COM_FIX404ERRORLINKS_ERROR_URL').':</strong> '.$mail_error_url_link;
			$mail_ip_country = "<strong>".JText::_('COM_FIX404ERRORLINKS_IP').":</strong> ".self::$ip_address." | <strong>".JText::_('COM_FIX404ERRORLINKS_COUNTRY').":</strong> ".self::$country;
			$mail_fix_this_error_url = '<a target="_blank" style="text-decoration: none; color: #FFFFFF; font-family: sans-serif; font-size: 17px; font-weight: 400; line-height: 120%; display: block;" href="'.self::$live_url.'administrator/index.php?option=com_fix404errorlinks&view=redirect&layout=edit&source_url='.self::$current_url_path.'">'.JText::_('COM_FIX404ERRORLINKS_FIX_THIS_ERROR_URL').'</a>';
			$website_link = str_replace('administrator', '', JURI::base());
			$mail_website_link = '<a href="'.$website_link.'" target="_blank" title="'.$website_link.'">'.$website_link.'</a>';
			$mail_media_dir = 'https://raw.githubusercontent.com/web357/cdn/master';
			$mail_small_logo_img = '<img border="0" vspace="0" hspace="0" src="'.$mail_media_dir.'/logo.png" width="64" height="64" alt="'.JText::_('COM_FIX404ERRORLINKS_ERROR_URL').'" title="'.JText::_('COM_FIX404ERRORLINKS_ERROR_URL').'" style=" color: #000000; font-size: 10px; margin: 0; padding: 0; outline: none; text-decoration: none; -ms-interpolation-mode: bicubic; border: none; display: block;" />';
			$error_details_txt = JText::_('COM_FIX404ERRORLINKS_ERROR_DETAILS') . ($is_danger ? ' ('.JText::_('COM_FIX404ERRORLINKS_INCLUDES_DANGER_STRING').')' : '');
			$error_details_img = '<img border="0" vspace="0" hspace="0" style="padding: 0; margin: 0; outline: none; text-decoration: none; -ms-interpolation-mode: bicubic; border: none; display: block; color: #000000;" src="'.$mail_media_dir.'/error-details.png" alt="'.JText::_('COM_FIX404ERRORLINKS_ERROR_DETAILS').'" title="'.JText::_('COM_FIX404ERRORLINKS_ERROR_DETAILS').'" width="64" height="64">';
			$user_details_txt = JText::_('COM_FIX404ERRORLINKS_USER_DETAILS') . (self::$is_bot ? ' ('.JText::_('COM_FIX404ERRORLINKS_IS_A_BOT').')' : '');
			$user_details_img = '<img border="0" vspace="0" hspace="0" style="padding: 0; margin: 0; outline: none; text-decoration: none; -ms-interpolation-mode: bicubic; border: none; display: block; color: #000000;" src="'.$mail_media_dir.'/user-details.png" alt="'.JText::_('COM_FIX404ERRORLINKS_USER_DETAILS').'" title="'.JText::_('COM_FIX404ERRORLINKS_USER_DETAILS').'" width="64" height="64">';
			$mail_component = (!empty($option)) ? "<strong>".JText::_('COM_FIX404ERRORLINKS_COMPONENT').":</strong> ".$option."<br>" : '';
			$mail_datetime = JHtml::_('date', self::$joomla_datetime, JText::_('COM_FIX404ERRORLINKS_DEFAULT_DATE_FORMAT'));
			$user_details_html = '';
			$user_details_html .= (self::$user_id) ? "<strong>".JText::_('COM_FIX404ERRORLINKS_USER_ID').":</strong> ".self::$user_id."<br>" : "";
			$user_details_html .= "<strong>".JText::_('COM_FIX404ERRORLINKS_IP_ADDRESS').":</strong> ".self::$ip_address."".(self::$is_bot ? ' ('.JText::_('COM_FIX404ERRORLINKS_BOT').')' : '') . ' <a href="http://myip.ms/info/whois/'.self::$ip_address.'" target="_blank"><small><em>('.JText::_('COM_FIX404ERRORLINKS_CHECK_WHOIS').')</em></small></a>'."<br>";
			$user_details_html .= "<strong>".JText::_('COM_FIX404ERRORLINKS_COUNTRY').":</strong> ".self::$country."<br>";
			$user_details_html .= (self::$is_bot == FALSE) ? "<strong>".JText::_('COM_FIX404ERRORLINKS_BROWSER').":</strong> ".self::$browser."<br>" : "";
			$user_details_html .= (self::$is_bot == FALSE) ? "<strong>".JText::_('COM_FIX404ERRORLINKS_OPERATING_SYSTEM').":</strong> ".self::$operating_system."<br>" : "";
			$user_details_html .= (self::$is_bot == TRUE) ? "<strong>".JText::_('COM_FIX404ERRORLINKS_USER_AGENT').":</strong> <small style=\"font-size:14px;\">".self::$user_agent."</small><br>" : "";
			$mail_referrer_link = '<small style="font-size:13px;"><a href="'.self::$referrer.'" title="'.self::$referrer.'" target="_blank">'.self::$referrer.'</a><small></small>';
			$user_details_html .= (!empty(self::$referrer)) ? "<strong>".JText::_('COM_FIX404ERRORLINKS_REFERRER').":</strong> ".$mail_referrer_link : "";
			$error_code_txt = JText::_('COM_FIX404ERRORLINKS_ERROR_CODE');
			$error_message_txt = JText::_('COM_FIX404ERRORLINKS_ERROR_MESSAGE');
			$error_url_txt = JText::_('COM_FIX404ERRORLINKS_ERROR_URL');
			$datetime_txt = JText::_('COM_FIX404ERRORLINKS_DATETIME');
			$hits_txt = JText::_('COM_FIX404ERRORLINKS_HITS');
			$footer_txt = JText::sprintf(JText::_('COM_FIX404ERRORLINKS_EMAIL_FOOTER_TEXT'), $mail_website_link);

			// Send direct email to Admin
			// build body for the email
			$html = <<<HTML
<html xmlns="http://www.w3.org/1999/xhtml">
<head>
<meta http-equiv="content-type" content="text/html; charset=utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0;">
<meta name="format-detection" content="telephone=no"/>

<style>
/* Reset styles */ 
body { margin: 0; padding: 0; min-width: 100%; width: 100% !important; height: 100% !important;}
body, table, td, div, p, a { -webkit-font-smoothing: antialiased; text-size-adjust: 100%; -ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; line-height: 100%; }
table, td { mso-table-lspace: 0pt; mso-table-rspace: 0pt; border-collapse: collapse !important; border-spacing: 0; }
img { border: 0; line-height: 100%; outline: none; text-decoration: none; -ms-interpolation-mode: bicubic; }
#outlook a { padding: 0; }
.ReadMsgBody { width: 100%; } .ExternalClass { width: 100%; }
.ExternalClass, .ExternalClass p, .ExternalClass span, .ExternalClass font, .ExternalClass td, .ExternalClass div { line-height: 100%; }
.mail_table_width { width: 600px; }
/* Rounded corners for advanced mail clients only */ 
@media all and (min-width: 760px) {
	.container { border-radius: 8px; -webkit-border-radius: 8px; -moz-border-radius: 8px; -khtml-border-radius: 8px;}
}

/* Set color for auto links (addresses, dates, etc.) */ 
a, a:hover {
	color: #127DB3;
}
.footer a, .footer a:hover {
	color: #999999;
}
</style>

<!-- MESSAGE SUBJECT -->
<title>{$subject}</title>

</head>

<!-- BODY -->
<body topmargin="0" rightmargin="0" bottommargin="0" leftmargin="0" marginwidth="0" marginheight="0" width="100%" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; width: 100%; height: 100%; -webkit-font-smoothing: antialiased; text-size-adjust: 100%; -ms-text-size-adjust: 100%; -webkit-text-size-adjust: 100%; line-height: 100%; background-color: #F0F0F0; color: #000000;" bgcolor="#F0F0F0" text="#000000">

	<table width="100%" align="center" border="0" cellpadding="0" cellspacing="0" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; width: 100%;">
		<tr>
			<td align="center" valign="top" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0;" bgcolor="#F0F0F0">
			<div class="mail_table_width">

					<!-- LOGO -->
					<table border="0" cellpadding="0" cellspacing="0" align="center" width="760" style="border-collapse: collapse; border-spacing: 0; padding: 0; width: inherit;max-width: 760px;">
						<tr>
							<td align="center" valign="top" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-left: 6.25%; padding-right: 6.25%; width: 87.5%; padding-top: 20px; padding-bottom: 20px;">
								{$mail_small_logo_img}
							</td>
						</tr>
					</table>

					<table border="0" cellpadding="0" cellspacing="0" align="center" bgcolor="#FFFFFF" width="760" style="border-collapse: collapse; border-spacing: 0; padding: 0; width: inherit; max-width: 760px;" class="container">

						<!-- EMAIL TITLE -->
						<tr>
							<td align="center" valign="top" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-left: 6.25%; padding-right: 6.25%; width: 87.5%; font-size: 24px; font-weight: bold; line-height: 130%; padding-top: 25px; color: #000000; font-family: sans-serif;">
								{$mail_subject}
							</td>
						</tr>
					
						<!-- ERROR URL, IP, COUNTRY -->
						<tr>
							<td align="center" valign="top" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-bottom: 3px; padding-left: 6.25%; padding-right: 6.25%; width: 87.5%; font-size: 18px; font-weight: 300; line-height: 180%; padding-top: 5px; color: #000000; font-family: sans-serif;">
								{$mail_error_url_txt}<br>
								{$mail_ip_country}
							</td>
						</tr>

						<!-- BUTTON -->
						<tr>
							<td align="center" valign="top" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-left: 6.25%; padding-right: 6.25%; width: 87.5%; padding-top: 10px; padding-bottom: 5px;">
								<table border="0" cellpadding="0" cellspacing="0" align="center" style="max-width: 240px; min-width: 120px; border-collapse: collapse; border-spacing: 0; padding: 0;">
									<tr>
										<td align="center" valign="middle" style="padding: 0; margin: 0; text-decoration: none; border-collapse: collapse; border-spacing: 0; border-radius: 4px; -webkit-border-radius: 4px; -moz-border-radius: 4px; -khtml-border-radius: 4px; padding: 12px 24px;" bgcolor="#10b065">
											{$mail_fix_this_error_url}
										</td>
									</tr>
								</table>
							</td>
						</tr>

						<!-- LINE -->
						<tr>	
							<td align="center" valign="top" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-left: 6.25%; padding-right: 6.25%; width: 87.5%; padding-top: 15px;"><hr color="#E0E0E0" align="center" width="100%" size="1" noshade style="margin: 0; padding: 0;" />
							</td>
						</tr>

						<!-- DETAILS -->
						<tr>
							<td align="center" valign="top" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-left: 6.25%; padding-right: 6.25%;">
							
								<table align="center" border="0" cellspacing="0" cellpadding="0" style="width: inherit; margin: 0; padding: 0; border-collapse: collapse; border-spacing: 0;">
								
									<!-- HEADING -->
									<tr>
										<td align="center" colspan="2" style="border-collapse: collapse; border-spacing: 0; padding-top: 20px;">
											<strong style="color: #333333; font-size: 18px; font-family: sans-serif;">{$error_details_txt}</strong>
										</td>
									</tr>

									<tr>
										<!-- IMAGE -->
										<td align="left" valign="top" style="border-collapse: collapse; border-spacing: 0; padding-top: 30px; padding-right: 20px;">{$error_details_img}</td>

										<!-- TEXT -->
										<td align="left" valign="top" style="font-size: 17px; font-weight: 400; line-height: 160%; border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-top: 25px; color: #000000; font-family: sans-serif;">
											<strong>{$error_code_txt}:</strong> {$get_error_code}<br>
											<strong>{$error_message_txt}:</strong> {$get_error_message}<br>
											<strong>{$error_url_txt}:</strong> <small style="font-size:13px;">{$mail_error_url_link}</small><br>
											{$mail_component}
											<strong>{$datetime_txt}:</strong> {$mail_datetime}<br>
											<strong>{$hits_txt}:</strong> {$hits_count}
										</td>
									</tr>

									<!-- HEADING -->
									<tr>
										<td align="center" colspan="2" style="border-collapse: collapse; border-spacing: 0; padding-top: 20px;">
											<hr color="#E0E0E0" align="center" width="100%" size="1" noshade style="margin: 0; padding: 0; margin-bottom: 20px;" />
											<strong style="color: #333333; font-size: 18px; font-family: sans-serif;">{$user_details_txt}</strong>
										</td>
									</tr>
									
									<tr>
										<!-- IMAGE -->
										<td align="left" valign="top" style="border-collapse: collapse; border-spacing: 0; padding-top: 10px; padding-right: 20px;">{$user_details_img}</td>

										<!-- TEXT -->
										<td align="left" valign="top" style="font-size: 17px; font-weight: 400; line-height: 160%; border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-top: 10px; padding-bottom: 25px; color: #000000; font-family: sans-serif;">
											{$user_details_html}
										</td>
									</tr>
								</table>
							</td>
						</tr>
					</table>

					<!-- FOOTER -->
					<table border="0" cellpadding="0" cellspacing="0" align="center" width="760" style="border-collapse: collapse; border-spacing: 0; padding: 0; width: inherit; max-width: 760px;">
						<tr>
							<td align="center" valign="top" style="border-collapse: collapse; border-spacing: 0; margin: 0; padding: 0; padding-left: 6.25%; padding-right: 6.25%; width: 87.5%; font-size: 13px; font-weight: 400; line-height: 150%; padding-top: 20px; padding-bottom: 20px; color: #999999; font-family: sans-serif;" class="footer">
								{$footer_txt}
							</td>
						</tr>
					</table>
				</div>
			</td>
		</tr>
	</table>
</body>
</html>
HTML;
			// get email vars before send
			$app = Factory::getApplication();
			$from = $app->get('mailfrom');
			$email_recipients = $comParams->get('email_recipient', $from);
			$to = explode("\n", $email_recipients);
			$body = $html;

			// Check if this IP is excluded from the extension functionalities (Not any logs and notifications from this IP address will be stored) 
			if (
				in_array(self::$continent_code, $exclude_continents) == FALSE && // Exclude Continents
				in_array(self::$country_code, $exclude_countries) == FALSE && // Exclude Countries
				(($email_notifications_from_guests == TRUE && (self::$is_bot == FALSE && $is_danger == FALSE)) || // guests
				($email_notifications_from_danger_strings == TRUE && $email_notifications_from_bots == TRUE && (self::$is_bot == TRUE && $is_danger == TRUE)) || // bot which includes danger string
				($email_notifications_from_danger_strings == TRUE && (self::$is_bot == FALSE && $is_danger == TRUE)) || // danger strings
				($email_notifications_from_bots == TRUE && (self::$is_bot == TRUE && $is_danger == FALSE))) // bots
				)
				{

				/**
				 * Mailsent
				 */
				if ($error_url->mailsent < 1) {
					$mailsent = true;
				}
				else
				{
					$mailsent = false;
				}

				/**
				 * Restrictions
				 */ 
				$send_email_a = true;
				$send_email_b = true;
				// check if the email notification has been already sent 
				$send_email_only_once = $comParams->get('send_email_only_once', '0');
				if ($send_email_only_once == true && $mailsent == true)
				{
					$send_email_a = true;
				}
				elseif ($send_email_only_once == true && $mailsent == false)
				{
					$send_email_a = false;
				}

				// Send email only if the Error link visited over X times
				$send_email_with_over_x_hits = $comParams->get('send_email_with_over_x_hits', '0');
				$send_email_with_over_x_hits_count = $comParams->get('send_email_with_over_x_hits_count', '5');
				if ($send_email_with_over_x_hits == true)
				{
					if ($error_url->hits >= $send_email_with_over_x_hits_count)
					{
						$send_email_b = true;
					}
					else
					{
						$send_email_b = false;
					}
				}

				// send mail
				if ($send_email_a == true && $send_email_b == true)
				{
					if ($error_url === null)
					{
						// Store the data into the database
						$error_url = new stdClass();
						$error_url->mailsent = 1;

						$db = Factory::getDbo();

						try
						{
							$db->insertObject('#__fix404errorlinks_error404logs', $error_url, 'id');
						}
						catch (Exception $e)
						{
							JErrorPage::render(new Exception(JText::_('Fix404ErrorLinks: Error Updating Database.'), 500, $e));
						}
					}
					else
					{
						$error_url->mailsent = 1;

						try
						{
							$db->updateObject('#__fix404errorlinks_error404logs', $error_url, 'error_url');
						}
						catch (Exception $e)
						{
							JErrorPage::render(new Exception(JText::_('Fix404ErrorLinks: Error Updating Database.'), 500, $e));
						}
					}

					self::sendMail($from, $to, $subject, $body);
				}
			}

			// Check if there are any redirects before the redirection
			self::redirectFixedURLs();

			// redirect danger urls to the right page
			if ($is_danger == TRUE && $danger_strings_redirection == TRUE)
			{
				if ($danger_strings_redirection_type == 'url' && !empty($danger_url_redirects_to_url))
				{
					self::redirectToTheFinalDestination($danger_url_redirects_to_url, self::$current_url_path, 404, 0, 'full_url', 'redirect_to_url', 301);
				}
				elseif ($danger_strings_redirection_type == 'menuitem' && !empty($danger_url_redirects_to_menuitem))
				{
					self::redirectToTheFinalDestination($danger_url_redirects_to_menuitem, self::$current_url_path, 404, 0, 'full_url', 'redirect_to_menu_item', 301);
				}
				else
				{
					self::redirectToTheFinalDestination('index.php');
				}
			}

			// redirect all 404 Error urls to the right page
			if ($redirect_all_404errors_in_one_page == TRUE)
			{
				if ($redirect_all_404errors_in_one_page_redirection_type == 'url' && !empty($all_404_error_urls_redirects_to_url))
				{
					self::redirectToTheFinalDestination($all_404_error_urls_redirects_to_url, self::$current_url_path, 404, 0, 'full_url', 'redirect_to_url', 301);
				}
				elseif ($redirect_all_404errors_in_one_page_redirection_type == 'menuitem' && !empty($all_404_error_urls_redirects_to_menuitem))
				{
					self::redirectToTheFinalDestination($all_404_error_urls_redirects_to_menuitem, self::$current_url_path, 404, 0, 'full_url', 'redirect_to_menu_item', 301);
				}
				else
				{
					self::redirectToTheFinalDestination('index.php');
				}
			}

			// Render the error page.
			if (version_compare(JVERSION, '4.0', 'ge')) {

				// set header
				$http_status_code_arr = self::getHttpStatusCodes();
				if (isset($http_status_code_arr[$error->getCode()]))
				{
					header($http_status_code_arr[$error->getCode()]);
				}

				// Render the custom 404 error page.
				self::renderCustom404ErrorPage($error);	
            }
			else
			{
				// Render the error page.
				JError::customErrorPage($error);
			}
		}
		else 
		{
			 self::redirectFixedURLs();
		}
	}

	/**
	 * Render the error page based on an exception.
	 *
	 * @param   \Exception|\Throwable  $error  An Exception or Throwable (PHP 7+) object for which to render the error page.
	 *
	 * @return  void
	 *
	 * @since   3.0
	 */
	public static function renderCustom404ErrorPage($error)
	{
		$expectedClass = PHP_MAJOR_VERSION >= 7 ? '\Throwable' : '\Exception';
		$isException   = $error instanceof $expectedClass;

		// In PHP 5, the $error object should be an instance of \Exception; PHP 7 should be a Throwable implementation
		if ($isException)
		{
			try
			{
				// Try to log the error, but don't let the logging cause a fatal error
				try
				{
					\JLog::add(
						sprintf(
							'Uncaught %1$s of type %2$s thrown. Stack trace: %3$s',
							$expectedClass,
							get_class($error),
							$error->getTraceAsString()
						),
						\JLog::CRITICAL,
						'error'
					);
				}
				catch (\Throwable $e)
				{
					// Logging failed, don't make a stink about it though
				}
				catch (\Exception $e)
				{
					// Logging failed, don't make a stink about it though
				}

				$app = \JFactory::getApplication();

				// If site is offline and it's a 404 error, just go to index (to see offline message, instead of 404)
				if ($error->getCode() == '404' && $app->get('offline') == 1)
				{
					$app->redirect('index.php');
				}

				$attributes = array(
					'charset'   => 'utf-8',
					'lineend'   => 'unix',
					'tab'       => "\t",
					'language'  => 'en-GB',
					'direction' => 'ltr',
				);

				// If there is a \JLanguage instance in \JFactory then let's pull the language and direction from its metadata
				if (\JFactory::$language)
				{
					$attributes['language']  = \JFactory::getLanguage()->getTag();
					$attributes['direction'] = \JFactory::getLanguage()->isRtl() ? 'rtl' : 'ltr';
				}

				$document = \JDocument::getInstance('error', $attributes);

				if (!$document)
				{
					// We're probably in an CLI environment
					jexit($error->getMessage());
				}

				return;
			}

			catch (\Throwable $e)
			{
				// Pass the error down
			}
			catch (\Exception $e)
			{
				// Pass the error down
			}
		}

		// This isn't an Exception, we can't handle it.
		if (!headers_sent())
		{
			header('HTTP/1.1 500 Internal Server Error');
		}

		$message = 'Error';

		if ($isException)
		{
			// Make sure we do not display sensitive data in production environments
			if (ini_get('display_errors'))
			{
				$message .= ': ';

				if (isset($e))
				{
					$message .= $e->getMessage() . ': ';
				}

				$message .= $error->getMessage();
			}
		}

		jexit(1);
	}

	public static function getHttpStatusCodes()
	{
		return array(
			100 => 'HTTP/1.1 100 Continue',
			101 => 'HTTP/1.1 101 Switching Protocols',
			102 => 'HTTP/1.1 102 Processing',
			200 => 'HTTP/1.1 200 OK',
			201 => 'HTTP/1.1 201 Created',
			202 => 'HTTP/1.1 202 Accepted',
			203 => 'HTTP/1.1 203 Non-Authoritative Information',
			204 => 'HTTP/1.1 204 No Content',
			205 => 'HTTP/1.1 205 Reset Content',
			206 => 'HTTP/1.1 206 Partial Content',
			207 => 'HTTP/1.1 207 Multi-Status',
			208 => 'HTTP/1.1 208 Already Reported',
			226 => 'HTTP/1.1 226 IM Used',
			300 => 'HTTP/1.1 300 Multiple Choices',
			301 => 'HTTP/1.1 301 Moved Permanently',
			302 => 'HTTP/1.1 302 Found',
			303 => 'HTTP/1.1 303 See other',
			304 => 'HTTP/1.1 304 Not Modified',
			305 => 'HTTP/1.1 305 Use Proxy',
			306 => 'HTTP/1.1 306 (Unused)',
			307 => 'HTTP/1.1 307 Temporary Redirect',
			308 => 'HTTP/1.1 308 Permanent Redirect',
			400 => 'HTTP/1.1 400 Bad Request',
			401 => 'HTTP/1.1 401 Unauthorized',
			402 => 'HTTP/1.1 402 Payment Required',
			403 => 'HTTP/1.1 403 Forbidden',
			404 => 'HTTP/1.1 404 Not Found',
			405 => 'HTTP/1.1 405 Method Not Allowed',
			406 => 'HTTP/1.1 406 Not Acceptable',
			407 => 'HTTP/1.1 407 Proxy Authentication Required',
			408 => 'HTTP/1.1 408 Request Timeout',
			409 => 'HTTP/1.1 409 Conflict',
			410 => 'HTTP/1.1 410 Gone',
			411 => 'HTTP/1.1 411 Length Required',
			412 => 'HTTP/1.1 412 Precondition Failed',
			413 => 'HTTP/1.1 413 Payload Too Large',
			414 => 'HTTP/1.1 414 URI Too Long',
			415 => 'HTTP/1.1 415 Unsupported Media Type',
			416 => 'HTTP/1.1 416 Requested Range Not Satisfiable',
			417 => 'HTTP/1.1 417 Expectation Failed',
			418 => 'HTTP/1.1 418 I\'m a teapot',
			422 => 'HTTP/1.1 422 Unprocessable Entity',
			423 => 'HTTP/1.1 423 Locked',
			424 => 'HTTP/1.1 424 Failed Dependency',
			425 => 'HTTP/1.1 425 Reserved for WebDAV advanced collections expired proposal',
			426 => 'HTTP/1.1 426 Upgrade Required',
			428 => 'HTTP/1.1 428 Precondition Required',
			429 => 'HTTP/1.1 429 Too Many Requests',
			431 => 'HTTP/1.1 431 Request Header Fields Too Large',
			451 => 'HTTP/1.1 451 Unavailable For Legal Reasons',
			500 => 'HTTP/1.1 500 Internal Server Error',
			501 => 'HTTP/1.1 501 Not Implemented',
			502 => 'HTTP/1.1 502 Bad Gateway',
			503 => 'HTTP/1.1 503 Service Unavailable',
			504 => 'HTTP/1.1 504 Gateway Timeout',
			505 => 'HTTP/1.1 505 HTTP Version Not Supported',
			506 => 'HTTP/1.1 506 Variant Also Negotiates (Experimental)',
			507 => 'HTTP/1.1 507 Insufficient Storage',
			508 => 'HTTP/1.1 508 Loop Detected',
			510 => 'HTTP/1.1 510 Not Extended',
			511 => 'HTTP/1.1 511 Network Authentication Required',
		);
	}

	public static function startsWith($haystack, $needle) 
	{
		// search backwards starting from haystack length characters from the end
		$haystack = str_replace('index.php/', '', $haystack);
		return $needle === "" || strrpos($haystack, $needle, -strlen($haystack)) !== FALSE;
	}
	
	public static function endsWith($haystack, $needle) 
	{
		// search forward starting from end minus needle length characters
		return $needle === "" || (($temp = strlen($haystack) - strlen($needle)) >= 0 && strpos($haystack, $needle, $temp) !== FALSE);
	}

	// send email
	public static function sendMail($from, $to, $subject, $body)
	{
		$app = Factory::getApplication();
		$sendMailEnabled = (bool) $app->get('mailonline', 1);

		if ($sendMailEnabled)
		{
			$mailer = Factory::getMailer();
			$from = (!empty($from)) ? $from : array($app->get('fromname'), $app->get('mailfrom'));
			$mailer->setSender($from);
			$mailer->setSubject($subject);
			$mailer->isHTML(true);
			$mailer->Encoding = 'base64';
			$mailer->setBody($body);
			$mailer->addRecipient($to);

			try
			{
				$mailer->Send();
			}
			catch (Exception $e)
			{
				// throw new Exception(JText::_('Error sending email: ' . $e->getMessage()), 403);
			}
		}

		return true;
	}

	/**
	 * Get User's Browser (e.g. Google Chrome (64.0.3282.186))
	 * @return string
	 */
	public static function getBrowser()
	{ 
		$bname = 'Unknown';
		$ub = 'Unknown';
		$platform = 'Unknown';
		$version= "";
	
		//First get the platform?
		if (preg_match('/linux/i', self::$user_agent)) {
			$platform = 'linux';
		}
		elseif (preg_match('/macintosh|mac os x/i', self::$user_agent)) {
			$platform = 'mac';
		}
		elseif (preg_match('/windows|win32/i', self::$user_agent)) {
			$platform = 'windows';
		}
	
		// Next get the name of the useragent yes seperately and for good reason
		if(preg_match('/MSIE/i',self::$user_agent) && !preg_match('/Opera/i',self::$user_agent)) 
		{ 
			$bname = 'Internet Explorer'; 
			$ub = "MSIE"; 
		} 
		elseif(preg_match('/Trident/i',self::$user_agent)) 
		{ // this condition is for IE11
			$bname = 'Internet Explorer'; 
			$ub = "rv"; 
		} 
		elseif(preg_match('/Firefox/i',self::$user_agent)) 
		{ 
			$bname = 'Mozilla Firefox'; 
			$ub = "Firefox"; 
		} 
		elseif(preg_match('/Chrome/i',self::$user_agent)) 
		{ 
			$bname = 'Google Chrome'; 
			$ub = "Chrome"; 
		} 
		elseif(preg_match('/Safari/i',self::$user_agent)) 
		{ 
			$bname = 'Apple Safari'; 
			$ub = "Safari"; 
		} 
		elseif(preg_match('/Opera/i',self::$user_agent)) 
		{ 
			$bname = 'Opera'; 
			$ub = "Opera"; 
		} 
		elseif(preg_match('/Netscape/i',self::$user_agent)) 
		{ 
			$bname = 'Netscape'; 
			$ub = "Netscape"; 
		} 
		
		// finally get the correct version number
		// Added "|:"
		$known = array('Version', $ub, 'other');
		$pattern = '#(?<browser>' . join('|', $known) .
		 ')[/|: ]+(?<version>[0-9.|a-zA-Z.]*)#';
		if (!preg_match_all($pattern, self::$user_agent, $matches)) {
			// we have no matching number just continue
		}
	
		// see how many we have
		$i = count($matches['browser']);
		if ($i != 1):
			//we will have two since we are not using 'other' argument yet
			//see if version is before or after the name
			if (strripos(self::$user_agent,"Version") < strripos(self::$user_agent,$ub)):
				$version= $matches['version'][0];
			else:
				if (isset($matches['version'][1])):
					$version = $matches['version'][1];
				elseif (isset($matches['version'][0])):
					$version= $matches['version'][0];
				else:
					$version = '';
				endif;
			endif;
		else:
			if (isset($matches['version'][0])):
				$version= $matches['version'][0];
			else:
				$version = '';
			endif;
		endif;

		// check if we have a number
		if ($version==null || $version=="") {$version="?";}
	
		$browser_details_arr = array(
			'userAgent' => self::$user_agent,
			'name'      => $bname,
			'version'   => $version,
			'platform'  => $platform,
			'pattern'    => $pattern
		);

		return $bname . ' (' . $version . ')';
	} 
	
	/**
	 * Get User's operating system (e.g. Windows 10 x64)
	 * @return string
	 */
	public static function getOS()
	{
		if (self::$is_bot)
		{
			return 'Unknown (bot)';
		}
		
		// https://stackoverflow.com/questions/18070154/get-operating-system-info-with-php
		$os_array = array(
			'windows nt 10'                              =>  'Windows 10',
			'windows nt 6.3'                             =>  'Windows 8.1',
			'windows nt 6.2'                             =>  'Windows 8',
			'windows nt 6.1|windows nt 7.0'              =>  'Windows 7',
			'windows nt 6.0'                             =>  'Windows Vista',
			'windows nt 5.2'                             =>  'Windows Server 2003/XP x64',
			'windows nt 5.1'                             =>  'Windows XP',
			'windows xp'                                 =>  'Windows XP',
			'windows nt 5.0|windows nt5.1|windows 2000'  =>  'Windows 2000',
			'windows me'                                 =>  'Windows ME',
			'windows nt 4.0|winnt4.0'                    =>  'Windows NT',
			'windows ce'                                 =>  'Windows CE',
			'windows 98|win98'                           =>  'Windows 98',
			'windows 95|win95'                           =>  'Windows 95',
			'win16'                                      =>  'Windows 3.11',
			'mac os x 10.1[^0-9]'                        =>  'Mac OS X Puma',
			'macintosh|mac os x'                         =>  'Mac OS X',
			'mac_powerpc'                                =>  'Mac OS 9',
			'linux'                                      =>  'Linux',
			'ubuntu'                                     =>  'Linux - Ubuntu',
			'iphone'                                     =>  'iPhone',
			'ipod'                                       =>  'iPod',
			'ipad'                                       =>  'iPad',
			'android'                                    =>  'Android',
			'blackberry'                                 =>  'BlackBerry',
			'webos'                                      =>  'Mobile',

			'(media center pc).([0-9]{1,2}\.[0-9]{1,2})'=>'Windows Media Center',
			'(win)([0-9]{1,2}\.[0-9x]{1,2})'=>'Windows',
			'(win)([0-9]{2})'=>'Windows',
			'(windows)([0-9x]{2})'=>'Windows',

			// Doesn't seem like these are necessary...not totally sure though..
			//'(winnt)([0-9]{1,2}\.[0-9]{1,2}){0,1}'=>'Windows NT',
			//'(windows nt)(([0-9]{1,2}\.[0-9]{1,2}){0,1})'=>'Windows NT', // fix by bg

			'Win 9x 4.90'=>'Windows ME',
			'(windows)([0-9]{1,2}\.[0-9]{1,2})'=>'Windows',
			'win32'=>'Windows',
			'(java)([0-9]{1,2}\.[0-9]{1,2}\.[0-9]{1,2})'=>'Java',
			'(Solaris)([0-9]{1,2}\.[0-9x]{1,2}){0,1}'=>'Solaris',
			'dos x86'=>'DOS',
			'Mac OS X'=>'Mac OS X',
			'Mac_PowerPC'=>'Macintosh PowerPC',
			'(mac|Macintosh)'=>'Mac OS',
			'(sunos)([0-9]{1,2}\.[0-9]{1,2}){0,1}'=>'SunOS',
			'(beos)([0-9]{1,2}\.[0-9]{1,2}){0,1}'=>'BeOS',
			'(risc os)([0-9]{1,2}\.[0-9]{1,2})'=>'RISC OS',
			'unix'=>'Unix',
			'os/2'=>'OS/2',
			'freebsd'=>'FreeBSD',
			'openbsd'=>'OpenBSD',
			'netbsd'=>'NetBSD',
			'irix'=>'IRIX',
			'plan9'=>'Plan9',
			'osf'=>'OSF',
			'aix'=>'AIX',
			'GNU Hurd'=>'GNU Hurd',
			'(fedora)'=>'Linux - Fedora',
			'(kubuntu)'=>'Linux - Kubuntu',
			'(ubuntu)'=>'Linux - Ubuntu',
			'(debian)'=>'Linux - Debian',
			'(CentOS)'=>'Linux - CentOS',
			'(Mandriva).([0-9]{1,3}(\.[0-9]{1,3})?(\.[0-9]{1,3})?)'=>'Linux - Mandriva',
			'(SUSE).([0-9]{1,3}(\.[0-9]{1,3})?(\.[0-9]{1,3})?)'=>'Linux - SUSE',
			'(Dropline)'=>'Linux - Slackware (Dropline GNOME)',
			'(ASPLinux)'=>'Linux - ASPLinux',
			'(Red Hat)'=>'Linux - Red Hat',
			// Loads of Linux machines will be detected as unix.
			// Actually, all of the linux machines I've checked have the 'X11' in the User Agent.
			//'X11'=>'Unix',
			'(linux)'=>'Linux',
			'(amigaos)([0-9]{1,2}\.[0-9]{1,2})'=>'AmigaOS',
			'amiga-aweb'=>'AmigaOS',
			'amiga'=>'Amiga',
			'AvantGo'=>'PalmOS',
			//'(Linux)([0-9]{1,2}\.[0-9]{1,2}\.[0-9]{1,3}(rel\.[0-9]{1,2}){0,1}-([0-9]{1,2}) i([0-9]{1})86){1}'=>'Linux',
			//'(Linux)([0-9]{1,2}\.[0-9]{1,2}\.[0-9]{1,3}(rel\.[0-9]{1,2}){0,1} i([0-9]{1}86)){1}'=>'Linux',
			//'(Linux)([0-9]{1,2}\.[0-9]{1,2}\.[0-9]{1,3}(rel\.[0-9]{1,2}){0,1})'=>'Linux',
			//'[0-9]{1,2}\.[0-9]{1,2}\.[0-9]{1,3})'=>'Linux',
			'(webtv)/([0-9]{1,2}\.[0-9]{1,2})'=>'WebTV',
			'Dreamcast'=>'Dreamcast OS',
			'GetRight'=>'Windows',
			'go!zilla'=>'Windows',
			'gozilla'=>'Windows',
			'gulliver'=>'Windows',
			'ia archiver'=>'Windows',
			'NetPositive'=>'Windows',
			'mass downloader'=>'Windows',
			'microsoft'=>'Windows',
			'offline explorer'=>'Windows',
			'teleport'=>'Windows',
			'web downloader'=>'Windows',
			'webcapture'=>'Windows',
			'webcollage'=>'Windows',
			'webcopier'=>'Windows',
			'webstripper'=>'Windows',
			'webzip'=>'Windows',
			'wget'=>'Windows',
			'Java'=>'Unknown',
			'flashget'=>'Windows',

			// delete next line if the script show not the right OS
			//'(PHP)/([0-9]{1,2}.[0-9]{1,2})'=>'PHP',
			'MS FrontPage'=>'Windows',
			'(msproxy)/([0-9]{1,2}.[0-9]{1,2})'=>'Windows',
			'(msie)([0-9]{1,2}.[0-9]{1,2})'=>'Windows',
			'libwww-perl'=>'Unix',
			'UP.Browser'=>'Windows CE',
			'NetAnts'=>'Windows'
		);

		// https://github.com/ahmad-sa3d/php-useragent/blob/master/core/user_agent.php
		$arch_regex = '/\b(x86_64|x86-64|Win64|WOW64|x64|ia64|amd64|ppc64|sparc64|IRIX64)\b/ix';
		$arch = preg_match($arch_regex, self::$user_agent) ? '64' : '32';

		foreach ($os_array as $regex => $value) 
		{
			if (preg_match('{\b('.$regex.')\b}i', self::$user_agent)) 
			{
				return $value.' (x'.$arch.')';
			}
		}

		return 'Unknown';
	}

	/**
	 * Gets data from an IP address
	 * 1. 'country_name' => Country Name (e.g. Greece)
	 * 2. 'country_code' => Country Code (e.g. GR)
	 * 3. 'continent' => Continent (e.g. EU) 
	 *
	 * @param   string  $ip      The IP address to look up
	 * @param   string  $type  country_name, country_code, continent.
	 *
	 * @return  mixed  A string with the name or code of the country, or the continent
	 */
	private function getDataFromGeoIP($type = 'country_name')
	{
		$result = 'XX';

		// require geoip2 library
		require_once(__DIR__ . '/lib/geoip2/geoip2.php');
		require_once(__DIR__ . '/lib/vendor/autoload.php');

		// If the GeoIP provider is not loaded return "XX" (no country detected)
		if (!class_exists('Web357GeoIp2'))
		{
			return $result;
		}

		// Use GeoIP to detect the country
		$geoip = new \Web357GeoIp2();

		switch ($type) {
			case 'country_name':
				$result = $geoip->getCountryName(self::$ip_address);
				break;
			case 'country_code':
				$result = $geoip->getCountryCode(self::$ip_address);
				break;
			case 'continent':
				$result = $geoip->getContinent(self::$ip_address);
				break;
			default:
				$result = $geoip->getCountryName(self::$ip_address);
				break;
		}

		// If detection failed, return "XX" (no country detected)
		if (empty($result))
		{
			$result = 'XX';
		}

		return $result;
	}

	public static function dangerStrings()
	{
		// Default Danger Strings
		return array(
			"phpmyadmin",
			"admin_area",
			"checkadmin",
			"webmaster", 
			"sysadm", 
		);
	}

	/**
     * Fix deprecated issues for idn_to_ascii
     *
     * @param [type] $url
     * @return void
     */
    public static function idnToAscii($url)
    {
        if (extension_loaded('intl')) {
            if (version_compare(phpversion(), '7.2', '>')) {
            if (defined('INTL_IDNA_VARIANT_UTS46')) {
                return (idn_to_ascii($url, 0, INTL_IDNA_VARIANT_UTS46) ?: $url);
                }
            }
            return (idn_to_ascii($url) ?: $url);
        }
        return $url;
	}
	
	public static function getUrlHttpCode($url)
	{
		return 404;
		$handle = curl_init($url);
		curl_setopt($handle,  CURLOPT_RETURNTRANSFER, TRUE);
		curl_setopt($handle, CURLOPT_SSL_VERIFYPEER, false);
		curl_setopt($handle, CURLOPT_SSL_VERIFYHOST, false);
		curl_setopt($handle, CURLOPT_SSLVERSION, 1);
		curl_setopt($handle, CURLOPT_SSL_CIPHER_LIST, 'TLSv1');

		/* Get the HTML or whatever is linked in $url. */
		$response = curl_exec($handle);

		/* Check for 404 (file not found). */
		$httpCode = curl_getinfo($handle, CURLINFO_HTTP_CODE);

		curl_close($handle);

		return (int) $httpCode;
	}

}