<?php
/**
 * @package DJ-Catalog2
 * @copyright Copyright (C) DJ-Extensions.com, All rights reserved.
 * @license http://www.gnu.org/licenses GNU/GPL
 * @author url: http://dj-extensions.com
 * @author email contact@dj-extensions.com
 */

use Joomla\Registry\Registry;

defined('_JEXEC') or die('Restricted access');


class Djcatalog2HelperPrice {
	
	static $priceRules;
	static $ordersCount;
	static $_labels = array();
	static $runOnce = array();
	
	static $exchangeRates = array();
	static $currencies;
	
	public static function calculate($price, $target, $tax_rule_id, $tax_rate = false, $user_id = null) {
		if ($tax_rate === false) {
			$tax_rate = self::getTaxRate($tax_rule_id, $user_id);
		}
		
		switch ($target) {
			case 'N' : {
				$price = self::getWithoutTax($price, $tax_rate);
				break;
			}
			case 'G' : {
				$price = self::getWithTax($price, $tax_rate);
				break;
			}
			case 'T' : {
				$price = self::getTax($price, $tax_rate);
				break;
			}
			default: break;
		}
		
		return $price;
	}
	
	public static function getTaxRate($tax_rule_id, $user_id = null) {
		$user = Djcatalog2Helper::getUserProfile($user_id);
		
		$tax_rate = 0;
		
		if (isset($user->tax_rules)) {
			if (isset($user->tax_rules[$tax_rule_id])) {
				$tax_rate=$user->tax_rules[$tax_rule_id];
			}
		}
		
		return $tax_rate;
	}
	
	public static function getWithoutTax($price, $tax_rate) {
		$tax = self::getTax($price, $tax_rate, true);
		$net = floatval($price) - $tax;
		return round($net, 2);
	}
	public static function getWithTax($price, $tax_rate) {
		$gross = floatval($price) + self::getTax($price, $tax_rate, false);
		return round($gross, 2);
	}
	public static function getTax($price, $tax_rate, $has_tax = null) {
		
		$params = JComponentHelper::getParams('com_djcatalog2');
		$default_has_tax = (bool)((int)$params->get('price_including_tax', 1) == 1);
		
		$tax = 0;
		if ($has_tax || ($has_tax === null) && $default_has_tax ) {
			$tax = floatval($price) * ($tax_rate / (100 + $tax_rate));
		} else if ($has_tax === false || ($has_tax === null && !$default_has_tax) ) {
			$tax = ($tax_rate * floatval($price)) / 100;
		}
		
		return round($tax, 2);
	}
	public static function getPrices($price, $old_price, $tax_rule_id, $tax_rate, $params){
		
		$prices = array('net'=>0, 'tax'=>0, 'gross'=>0, 'old'=>0, 'display'=>0, 'display2nd'=>false);
		
		$tax_already_incl = (bool)($params->get('price_including_tax', 1) == 1);
		$price_display = $params->get('price_display', 0);
		
		$user_currency_id = JFactory::getApplication()->getUserState('com_djcatalog2.checkout.currency');
		if ($user_currency_id > 0) {
			$price = static::currencyExchange($price);
			$old_price = static::currencyExchange($old_price);
		}
		
		if ($tax_already_incl) {
			$prices['gross']	= $price;
			$prices['net'] 		= self::calculate($price, 'N', $tax_rule_id, $tax_rate);
			$prices['tax'] 		= self::calculate($price, 'T', $tax_rule_id, $tax_rate);
			
			$prices['old_gross']	= $old_price;
			$prices['old_net'] 		= self::calculate($old_price, 'N', $tax_rule_id, $tax_rate);
		} else {
			$prices['net'] 		= $price;
			$prices['gross'] 	= self::calculate($price, 'G', $tax_rule_id, $tax_rate);
			$prices['tax'] 		= self::calculate($price, 'T', $tax_rule_id, $tax_rate);
			
			$prices['old_net'] 		= $old_price;
			$prices['old_gross'] 	= self::calculate($old_price, 'G', $tax_rule_id, $tax_rate);
		}
		
		if ($price_display == 0) {
			$prices['display'] = $prices['gross'];
			$prices['old_display'] = $prices['old_gross'];
		} else if ($price_display == 2) {
			$prices['display'] = $prices['gross'];
			$prices['display2nd'] = $prices['net'];
			$prices['old_display'] = $prices['old_gross'];
		} else {
			$prices['display'] 	= $prices['net'];
			$prices['old_display'] = $prices['old_net'];
		}
		
		return $prices;
	}
	
	public static function getCartPrices($price, $old_price, $tax_rule_id, $tax_rate, $quantity, $params){
		
		$prices_base = self::getPrices($price, $old_price, $tax_rule_id, $tax_rate, $params);
		$prices_total = self::getPrices(($price * $quantity), ($old_price * $quantity), $tax_rule_id, $tax_rate, $params);
		
		$prices = array('base' => $prices_base, 'total' => $prices_total);
		return $prices;
	}
	
	public static function currencyExchange($base_price) {
		$trade_price = $base_price;
		
		$user_currency_id = JFactory::getApplication()->getUserState('com_djcatalog2.checkout.currency');
		
		if ((int)$user_currency_id > 0) {
			$user_currency = static::getCurrencyById($user_currency_id);
			if (!empty($user_currency) && $user_currency->is_default == 0) {
				$latestRate = static::getExchangeRate($user_currency_id);
				if (!empty($latestRate) && $latestRate->target_rate != 0.0000) {
					$trade_price = $base_price / $latestRate->target_rate;
				}
			}
		}
		
		return $trade_price;
	}
	
	protected static function getExchangeRate($quote_id) {
		if (!isset(static::$exchangeRates[$quote_id])) {
			$db = JFactory::getDbo();
			$query = $db->getQuery(true);
			$query->select('e.*');
			$query->from('#__djc2_currency_exchange AS e');
			$query->join('INNER', '#__djc2_currencies AS c ON c.id = e.target_id');
			$query->where('base_id=' . (int)$quote_id);
			$query->where('c.is_default=1');
			$query->order('rate_date DESC');
			$db->setQuery($query);
			static::$exchangeRates[$quote_id] = $db->loadObject();
		}
		
		return static::$exchangeRates[$quote_id];
	}
	
	public static function getCurrencies() {
		if (!isset(static::$currencies)) {
			$db = JFactory::getDbo();
			$query = $db->getQuery(true);
			
			$query->select('id, name, currency, symbol, is_default');
			$query->from('#__djc2_currencies');
			$query->where('is_default=1');
			
			$uQuery = $db->getQuery(true);
			$uQuery->select('c2.id, c2.name, c2.currency, c2.symbol, c2.is_default');
			$uQuery->from('#__djc2_currencies AS c2');
			$uQuery->join('INNER', '#__djc2_currency_exchange AS e2 ON e2.base_id=c2.id');
			$uQuery->join('INNER', '#__djc2_currencies AS c3 ON c3.id=e2.target_id');
			$uQuery->where('c3.is_default=1');
			$uQuery->where('c2.is_default=0 AND DATE(e2.rate_date) <= DATE(' . $db->quote(JFactory::getDate()->toSql()).') ');
			$uQuery->group('c2.id, c2.name, c2.currency, c2.symbol');
			
			$query->unionAll($uQuery);
			
			$db->setQuery($query);
			static::$currencies = $db->loadObjectList('id');
		}
		
		return static::$currencies;
	}
	
	public static function getCurrencyById($currency_id) {
		$currencies = static::getCurrencies();
		if (isset($currencies[$currency_id])) {
			return $currencies[$currency_id];
		}
		
		return false;
	}
	
	public static function getCurrencyDefault() {
		$currencies = static::getCurrencies();
		foreach ($currencies as $currency) {
			if ($currency->is_default) {
				return $currency;
			}
		}
		
		return false;
	}
	
	public static function getCurrencyByCode($currency_code) {
		$currencies = static::getCurrencies();
		foreach ($currencies as $currency) {
			if ($currency->currency == $currency_code) {
				return $currency;
			}
		}
		
		return false;
	}
	
	public static function triggerCurrencyExAPI() {
		JPluginHelper::importPlugin('djcatalog2currency');
		JFactory::getApplication()->triggerEvent('onDJCatalog2FetchExchangeRates');
	}
	
	public static function getTierDiscounts($price, $tier_prices, $modifier='0', $quantity=1) {
		
		$result = array(
			'price' => $price,
			'discount' => 0,
			'discount_percent' => 0
		);
		
		if ($modifier == '0' || count($tier_prices) == 0) {
			return $result;
		}
		
		$tier_price = 0;
		
		for ($i = count($tier_prices) - 1; $i >= 0; $i--) {
			$tier = $tier_prices[$i];
			if ($quantity >= $tier->quantity && $tier->quantity > 0) {
				$tier_price = $tier->price;
				break;
			}
		}
		
		if ($tier_price == 0) {
			return $result;
		}
		
		switch ($modifier) {
			// by value
			case 'v': {
				$new_price = $price - $tier_price;
				break;
			}
			// by percent
			case 'p': {
				$new_price = round($price - ($price * $tier_price/100), 2);
				break;
			}
			// fixed value
			case 'f':
			default: {
				$new_price = $tier_price;
				break;
			}
		}
		
		if ($new_price <= 0 || $new_price >= $price) {
			return $result;
		}
		
		$result['price'] = $new_price;
		$result['discount'] = $price - $new_price;
		$result['discount_percent'] = round(100 * ($result['discount'] / $price), 2);
		
		return $result;
	}
	
	public static function applyPriceRules($price, $isCurrent, $context, $item = null, $item_type = null, $targetUser = null, $cart = null) {
		if (!in_array($context, array('each_item', 'grand_total', 'total_items', 'delivery', 'payment'))) {
			return $price;
		}
		
		$rules = static::getPriceRules($context);
		
		if (count($rules) < 1) {
			return $price;
		}
		
		$user = (is_null($targetUser)) ? Djcatalog2Helper::getUserProfile(JFactory::getApplication()->getUserState('com_djcatalog2.checkout.user_id', null)) : $targetUser;
		
		foreach ($rules as $rule) {
			$old_price = $price;
			$price = static::applyPriceRule($price, $isCurrent, $context, $rule, $item, $item_type, $user, $cart);
			if ($cart != null && is_array($price) && is_array($old_price)) {
				$operator = $rule->operation == 'sub' ? -1 : 1;
				$diffPrice = array(
					'net' => abs($old_price['net'] - $price['net']) * $operator,
					'tax' => abs($old_price['tax'] - $price['tax']) * $operator,
					'gross' => abs($old_price['gross'] - $price['gross']) * $operator
				);
				if ($diffPrice['gross'] != 0.0 || $diffPrice['net'] != 0.0 || $diffPrice['tax'] != 0.0) {
					$cart->addPriceComponent($rule, $diffPrice, 'rule');
				}
				//$cart->addPriceComponent($rule, $diffPrice, 'rule');
			}
		}
		
		return $price;
	}
	
	public static function applyCartPriceRules($cart, $price, $isCurrent, $context){
		return static::applyPriceRules($price, $isCurrent, $context, null, null, null, $cart);
	}
	
	public static function applyRulesBeforeInit(&$cart, &$items) {
		if (empty($items)) return;
		
		if (isset(static::$runOnce['beforeInit'])) return;
		static::$runOnce['beforeInit'] = true;
		
		$rules = static::getPriceRules('buy_x_discount');
		if (empty($rules)) return;
		
		$params = JComponentHelper::getParams('com_djcatalog2');
		
		foreach($rules as $rule) {
			if ((int)$rule->buy_x_count <= 0) continue;
			$buyXDiscount = (int)$rule->buy_x_count;
			
			$tempItems = [];
			foreach($items as $key => $item) {
				if (static::validatePriceRule($item->final_price, true, 'buy_x_discount', $rule, $item, 'item')) {
					$tempItems[$key] = $item;
				}
			}
			if (empty($tempItems)) continue;
			
			$tempCount = 0;
			$tempPrices = array();
			foreach($tempItems as $k=>$item) {
				$tempCount += $item->_quantity;
				$tempPrices[$k] = $item->final_price;
			}
			if ($tempCount >= $buyXDiscount) {
				asort($tempPrices);
				$tempDiscounted = floor($tempCount / ($buyXDiscount+1));
				$newNullPrice = max(0, $rule->value);
				while ($tempDiscounted > 0) {
					foreach ($tempPrices as $id => $price) {
						foreach($items as $k => &$item) {
							if (!empty($item->id) && $k == $id) {
								$quantity = (isset($cart->quantities[$k])) ? $cart->quantities[$k] : 1;
								$toDeduct = ($quantity >= $tempDiscounted) ? $tempDiscounted : $quantity;
								$item->_quantity = $quantity;
								
								if ($toDeduct > 0) {
									$item->_old_prices = static::getCartPrices($item->final_price, $item->price, $item->tax_rule_id, false,  $item->_quantity, $params);
									
									$oldFinal = $item->final_price;
									//$beforePrice = $oldFinal * $quantity;
									$afterPrice = $oldFinal * ($quantity - $toDeduct);
									$newUnitPrice = round($afterPrice/$quantity, 2);
									
									if ($newUnitPrice == 0) {
										$newUnitPrice = $newNullPrice;
									}
									
									if ($item->special_price > 0) {
										$item->special_price = $oldFinal;
									} else {
										$item->price = $oldFinal;
									}
									
									$item->_prices = static::getCartPrices($newUnitPrice, $item->price, $item->tax_rule_id, false,  $item->_quantity, $params);
									//$this->discount_value += ($beforePrice - $afterPrice);
									$tempDiscounted -= $toDeduct;
								}
							}
						}
						unset($item);
					}
				}
			}
			
		}
	}
	
	public static function applyRulesAfterInit(&$cart, &$items) {
		if (empty($items)) return;
		
		$mp_rules = static::getPriceRules('min_price');
		//$pd_rules = static::getPriceRules('after_delivery_payment');
		
		$params = JComponentHelper::getParams('com_djcatalog2');
		$tax_already_incl = (bool)($params->get('price_including_tax', 1) == 1);
		
		
		if (!empty($mp_rules) && empty(static::$runOnce['afterInitMinPrice'])) {
			foreach($mp_rules as $rule) {
				$tempItems = [];
				$tempTotal = 0;
				foreach($items as $key => $item) {
					if ($tax_already_incl) {
						$finalPrice = $item->_prices['base']['gross'];
					} else {
						$finalPrice = $item->_prices['base']['net'];
					}
					
					if (static::validatePriceRule($finalPrice, true, 'min_price', $rule, $item, 'item', null, $cart)) {
						$item->_tmp_final_price = $finalPrice;
						$tempItems[$key] = $item;
						$tempTotal += $item->_quantity * $finalPrice;
					}
				}
				if (empty($tempItems) || $tempTotal == 0.0) continue;
				if ($tempTotal >= $rule->min_price) {
					foreach($tempItems as $key => $item) {
						$oldPrice = $item->price;
						$items[$key]->_old_prices = Djcatalog2HelperPrice::getCartPrices($oldPrice, $item->price, $item->tax_rule_id, false,  $item->_quantity, $params);
						
						$items[$key]->final_price = static::applyOperation($item->_tmp_final_price, $rule->operation, $rule->value, $rule->type);
						$items[$key]->_prices = Djcatalog2HelperPrice::getCartPrices($items[$key]->final_price, $oldPrice, $item->tax_rule_id, false,  $item->_quantity, $params);
						unset($items[$key]->_tmp_final_price);
					}
				}
			}
			
			static::$runOnce['afterInitMinPrice'] = true;
		}
		
		/*
		 if (!empty($pd_rules) && !empty($cart->delivery) && !empty($cart->payment)  && !isset(static::$runOnce['afterInitAfterDeliveryPayment'])) {
		 foreach($pd_rules as $rule) {
		 
		 if (!empty($rule->deliveries) && !in_array($cart->delivery->id, $rule->deliveries)) {
		 continue;
		 }
		 if (!empty($rule->payments) && !in_array($cart->payment->id, $rule->payments)) {
		 continue;
		 }
		 
		 foreach($items as $key => $item) {
		 if ($tax_already_incl) {
		 $finalPrice = $item->_prices['base']['gross'];
		 } else {
		 $finalPrice = $item->_prices['base']['net'];
		 }
		 
		 if (false == static::validatePriceRule($finalPrice, true, 'after_delivery_payment', $rule, $item, 'item')) {
		 continue;
		 }
		 
		 $item->_tmp_final_price = $finalPrice;
		 
		 $oldPrice = $item->price;
		 $items[$key]->_old_prices = Djcatalog2HelperPrice::getCartPrices($oldPrice, $item->price, $item->tax_rule_id, false,  $item->_quantity, $params);
		 
		 $items[$key]->final_price = static::applyOperation($item->_tmp_final_price, $rule->operation, $rule->value, $rule->type);
		 $items[$key]->_prices = Djcatalog2HelperPrice::getCartPrices($items[$key]->final_price, $oldPrice, $item->tax_rule_id, false,  $item->_quantity, $params);
		 unset($items[$key]->_tmp_final_price);
		 }
		 }
		 
		 static::$runOnce['afterInitAfterDeliveryPayment'] = true;
		 }*/
	}
	
	public static function validatePriceRule($price, $isCurrent, $context, $rule, $item = null, $item_type = null, $targetUser = null, $cart = null) {
		if (!$isCurrent && !$rule->change_old_price && $rule->calc_type == 'each_item' ) {
			return false;
		}
		
		/* User Rule */
		if ($rule->user_id > 0) {
			if (empty($targetUser)) {
				return false;
			} else if ($rule->user_id != $targetUser->user_id) {
				return false;
			}
		}
		
		if (is_array($rule->customer_groups)  && count($rule->customer_groups)) {
			if (empty($targetUser)) {
				return false;
			} else if (empty($targetUser->customer_group_id)) {
				return false;
			} else {
				if (!in_array($targetUser->customer_group_id, $rule->customer_groups)){
					return false;
				}
			}
		}
		
		/* Country rule */
		if (!empty($rule->countries) && is_array($rule->countries)) {
			if (!in_array($targetUser->country_id, $rule->countries)) {
				return false;
			}
		}
		
		/* Zip Code rule*/
		if (!empty($rule->postcodes) && is_array($rule->postcodes)) {
			if (!isset($targetUser->postcode) || trim($targetUser->postcode) == '') {
				return false;
			}
			
			if (count($rule->postcodes) == 1) {
				if ($targetUser->postcode != $rule->postcodes[0]) {
					return false;
				}
			} else {
				if ( !($targetUser->postcode >= $rule->postcodes[0] && $targetUser->postcode <= $rule->postcodes[1])) {
					return false;
				}
			}
		}
		
		if (!empty($rule->deliveries)) {
			if (empty($cart)  || empty($cart->delivery)) {
				return false;
			}
			
			if (!in_array($cart->delivery->id, $rule->deliveries)) {
				return false;
			}
		}
		
		if (!empty($rule->payments)) {
			if (empty($cart) || empty($cart->payment)) {
				return false;
			}
			
			if (!in_array($cart->payment->id, $rule->payments)) {
				return false;
			}
		}
		
		if ($rule->min_orders != 0 || $rule->max_orders != -1) {
			if ($targetUser->user_id == 0) {
				return false;
			} else {
				$count = static::getUserOrders($targetUser, array('status' => array('C', 'P', 'F')));
				
				if ($rule->min_orders > 0 && $rule->min_orders > $count) {
					return false;
				}
				if ($rule->max_orders > -1 && $rule->max_orders < $count) {
					return false;
				}
			}
		}
		
		if (!empty($item)) {
			/* Product specific rules */
			if ($item_type == 'item') {
				if (!empty($rule->product_id) && is_array($rule->product_id)) {
					if (!in_array($item->id, $rule->product_id)) {
						return false;
					}
				}
				//else {
				if (!empty($rule->category_id) && is_array($rule->category_id)) {
					if (!in_array($item->cat_id, $rule->category_id)) {
						return false;
					}
				}
				
				if (!empty($rule->producer_id) && is_array($rule->producer_id)) {
					if (!in_array($item->producer_id, $rule->producer_id)) {
						return false;
					}
				}
				
				if (!empty($rule->excluded_product_id) && is_array($rule->excluded_product_id)) {
					if (in_array($item->id, $rule->excluded_product_id)) {
						return false;
					}
				}
				
				//Fetch labels
				if(isset($item->id) && $item->id) {
					static::applyPriceRuleLabels($rule, $item->id);
				}
				
				
				if (!empty($rule->extra_fields)) {
					//if (empty($item->_cart_features)) return false;
					foreach($rule->extra_fields as $efkey => $field) {
						$passed = false;
						if (!empty($item->_cart_features)) {
							foreach($field['option_id'] as $option_id) {
								if (in_array($option_id, $item->_cart_features)) {
									$passed = true;
									break;
								}
							}
						} else if (!empty($item->_extra_fields)){
							foreach($field['option_id'] as $option_id) {
								foreach($item->_extra_fields as $extra_field) {
									if (is_array($extra_field)) {
										foreach($extra_field as $option_attr) {
											if ($option_attr->option_id == $option_id) {
												$passed = true;
												break;
											}
										}
									}
								}
							}
						}
						
						if (!$passed) {
							return false;
						}
					}
				}
			}
		}
		
		return true;
	}
	
	public static function applyPriceRule($price, $isCurrent, $context, $rule, $item = null, $item_type = null, $targetUser = null, $cart = null) {
		if (static::validatePriceRule($price, $isCurrent, $context, $rule, $item, $item_type, $targetUser, $cart) == false) {
			return $price;
		}
		
		if (is_array($price)) {
			return static::applyOperationArr($price, $rule->operation, $rule->value, $rule->type);
		} else {
			return static::applyOperation($price, $rule->operation, $rule->value, $rule->type);
		}
	}
	
	public static function applyOperation($price, $operation, $value, $type) {
		$newPrice = $price;
		
		$value = $operation == 'sub' ? -1 * $value : $value;
		
		if ($type == 'percent') {
			$newPrice = $price + ($price * ($value / 100));
		} else if ($type == 'amount') {
			$newPrice += $value;
		}
		
		return max(0.0, $newPrice);
	}
	
	public static function applyOperationArr($price, $operation, $value, $type) {
		$newPrice = $price;
		foreach($price as $k=>$v) {
			$newPrice[$k] = static::applyOperation($v, $operation, $value, $type);
		}
		return $newPrice;
	}
	
	protected static function getPriceRules($context = null) {
		if (!isset(static::$priceRules)) {
			
			static::$priceRules = array(
				'each_item' => array(),
				'grand_total' => array(),
				'total_items' => array(),
				'payment' => array(),
				'delivery' => array(),
				'min_price' => array(),
				'buy_x_discount' => array(),
				'after_delivery_payment' => array(),
			);
			
			$db = JFactory::getDbo();
			
			$nowDate = JFactory::getDate()->toSql();
			$nullDate = $db->getNullDate();
			
			$query = $db->getQuery(true);
			$query->select('*');
			$query->from('#__djc2_price_rules');
			$query->order('ordering');
			$query->where('published=1');
			
			$query->where('(start_date = ' . $db->quote($nullDate).' OR start_date <= '.$db->quote($nowDate).')');
			$query->where('(expire_date = ' . $db->quote($nullDate).' OR expire_date >= '.$db->quote($nowDate).')');
			
			$db->setQuery($query);
			
			$rules = $db->loadObjectList();
			if (count($rules) > 0) {
				foreach ($rules as $rule) {
					
					if (!empty($rule->product_id)) {
						$registry = new Registry();
						$registry->loadString($rule->product_id);
						$temp = (array)$registry->toArray();
						$rule->product_id = array();
						for ($i=0; $i < count($temp); $i++) {
							$rule->product_id[] = $temp[$i];
						}
					} else {
						$rule->product_id = array();
					}
					
					if (!empty($rule->category_id)) {
						$registry = new Registry();
						$registry->loadString($rule->category_id);
						$temp = (array)$registry->toArray();
						$rule->category_id = array();
						for ($i=0; $i < count($temp); $i++) {
							$rule->category_id[] = $temp[$i];
						}
					} else {
						$rule->category_id = array();
					}
					
					if (!empty($rule->producer_id)) {
						$registry = new Registry();
						$registry->loadString($rule->producer_id);
						$temp = (array)$registry->toArray();
						$rule->producer_id = array();
						for ($i=0; $i < count($temp); $i++) {
							$rule->producer_id[] = $temp[$i];
						}
					} else {
						$rule->producer_id = array();
					}
					
					if (!empty($rule->countries)) {
						$registry = new Registry();
						$registry->loadString($rule->countries);
						$temp = (array)$registry->toArray();
						$rule->countries = array();
						for ($i=0; $i < count($temp); $i++) {
							$rule->countries[] = $temp[$i];
						}
					} else {
						$rule->countries = array();
					}
					
					if (!empty($rule->deliveries)) {
						$registry = new Registry();
						$registry->loadString($rule->deliveries);
						$temp = (array)$registry->toArray();
						$rule->deliveries = array();
						for ($i=0; $i < count($temp); $i++) {
							$rule->deliveries[] = $temp[$i];
						}
					} else {
						$rule->deliveries = array();
					}
					
					if (!empty($rule->payments)) {
						$registry = new Registry();
						$registry->loadString($rule->payments);
						$temp = (array)$registry->toArray();
						$rule->payments = array();
						for ($i=0; $i < count($temp); $i++) {
							$rule->payments[] = $temp[$i];
						}
					} else {
						$rule->payments = array();
					}
					
					if ($rule->postcodes != '' && is_string($rule->postcodes)) {
						$rule->postcodes = explode('|', $rule->postcodes);
					}
					if ($rule->excluded_product_id != '' && is_string($rule->excluded_product_id)) {
						$rule->excluded_product_id = explode(',', $rule->excluded_product_id);
					}
					if (is_string($rule->extra_fields) && trim($rule->extra_fields) != '') {
						$extra_fields = json_decode($rule->extra_fields, true);
						$rule->extra_fields = ($extra_fields) ? $extra_fields : null;
					}
					
					if (is_string($rule->extra_fields) && trim($rule->extra_fields) != '') {
						$extra_fields = json_decode($rule->extra_fields, true);
						$rule->extra_fields = ($extra_fields) ? $extra_fields : null;
					}
					
					if (is_string($rule->customer_groups) && trim($rule->customer_groups) != '') {
						$groups = json_decode($rule->customer_groups, true);
						$rule->customer_groups = ($groups) ? $groups : null;
					}
					
					static::$priceRules[$rule->calc_type][] = $rule;
				}
			}
		}
		
		return $context ? static::$priceRules[$context] : static::$priceRules;
	}
	
	public static function getUserOrders($user, $filters = array()) {
		if (!$user->user_id) {
			return 0;
		}
		
		$hash = md5($user->user_id.':'.serialize($filters));
		if (isset(static::$ordersCount[$hash])) {
			return static::$ordersCount[$hash];
		}
		
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);
		$query->select('COUNT(*)')->from('#__djc2_orders')->where('user_id='.(int)$user->user_id);
		if (isset($filters['status']) && is_array($filters['status'])) {
			$statuses = array();
			foreach($filters['status'] as $status) {
				$statuses[] = $db->quoteName('status').' = ' .$db->quote($status);
			}
			if (!empty($statuses)) {
				$query->where('(' . implode(' OR ', $statuses) . ')');
			}
		}
		$db->setQuery($query);
		
		static::$ordersCount[$hash] = $db->loadResult();
		return static::$ordersCount[$hash];
	}
	
	public static function applyPriceRuleLabels($rule, $itemId) {
		$labels = ((isset(self::$_labels[$itemId])) ? self::$_labels[$itemId] : array());
		
		$db = JFactory::getDbo();
		$query = $db->getQuery(true);
		
		$query
		->select('l.label, l.id')
		->from($db->quoteName('#__djc2_labels', 'l'))
		->join('INNER', $db->quoteName('#__djc2_price_rules_labels', 'p') . ' ON p.label_id = l.id')
		->where('p.price_rule_id = ' .  (int) $rule->id)
		->where('l.published = 1');
		
		$db->setQuery($query);
		$ruleLabels = $db->loadAssocList();
		
		$merged = array_merge($labels, $ruleLabels);
		$tempArr = array_unique(array_column($merged, 'id'));
		
		self::$_labels[$itemId] = array_intersect_key($merged, $tempArr);
	}
	
	public static function getItemPriceRuleLabels($itemId) {
		$labels = ((isset(self::$_labels[$itemId])) ? self::$_labels[$itemId] : array());
		return $labels;
	}
}
